// ****************************************************************************
// 
// Center Cut GUI
// Copyright (C) 2006-2010 J.D. Purcell
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************

using System;
using System.Globalization;
using System.IO;
using System.Threading;
using System.Windows.Forms;

namespace JDP {
	public partial class frmCenterCutGUI : Form {
		private int _fileIndex, _fileCount;
		private volatile bool _stopFlag;

		public frmCenterCutGUI() {
			InitializeComponent();
			General.SetFontAndScaling(this);

			string[] writerExts = AudioReadWrite.WriterExtensions;
			rbFLAC.Visible    = Array.Exists(writerExts, s => s == ".flac");
			rbWavPack.Visible = Array.Exists(writerExts, s => s == ".wv"  );
		}

		private bool Progress(double percent) {
			int n = (int)(percent * 100.0);

			if (n < 0) n = 0;
			if (n > 100) n = 100;

			BeginInvoke((MethodInvoker)(() => {
				lblStatus.Text = String.Format("File {0}/{1}: {2}%", _fileIndex + 1, _fileCount, n);
			}));

			return _stopFlag;
		}

		private void ProcessFiles(object parameters) {
			ProcessFilesParameters p = (ProcessFilesParameters)parameters;

			_fileCount = p.Paths.Length;

			for (_fileIndex = 0; _fileIndex < _fileCount; _fileIndex++) {
				string pathSrc = p.Paths[_fileIndex];
				string pathDstBase = Path.Combine(Path.GetDirectoryName(pathSrc), Path.GetFileNameWithoutExtension(pathSrc));
				string pathDstSides = null;
				string pathDstCenter = null;

				if (p.OutputSides) {
					pathDstSides = pathDstBase + p.SuffixSides + p.FormatExt;
				}
				if (p.OutputCenter) {
					pathDstCenter = pathDstBase + p.SuffixCenter + p.FormatExt;
				}

				try {
					CenterCutHelper.RunCenterCut(pathSrc, pathDstSides, pathDstCenter, p.DualCenter, p.BassToSides, p.AmpFactor, Progress);
				}
				catch (Exception ex) {
					Invoke((MethodInvoker)(() => {
						MessageBox.Show(ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
					}));
					break;
				}

				if (_stopFlag) break;
			}

			BeginInvoke((MethodInvoker)(() => {
				lblStatus.Text = String.Empty;
				btnRun.Text = "Run";
			}));
		}

		private void ListBoxSwapItems(ListBox listBox, int a, int b) {
			object temp = listBox.Items[a];
			listBox.Items[a] = listBox.Items[b];
			listBox.Items[b] = temp;
		}

		private void LoadSettings() {
			SettingsReader sr = new SettingsReader("Center Cut GUI", "settings.txt");
			string val;

			if ((val = sr.Load("Channels")) != null) {
				if (val == "Sides") {
					rbSides.Checked = true;
				}
				else if (val == "Center") {
					rbCenter.Checked = true;
				}
				else {
					rbBoth.Checked = true;
				}
			}

			if ((val = sr.Load("DualCenter")) != null) {
				chkDualCenter.Checked = (val == "1");
			}

			if ((val = sr.Load("BassToSides")) != null) {
				chkBassToSides.Checked = (val == "1");
			}

			if ((val = sr.Load("Amplitude")) != null) {
				double x;
				if (Double.TryParse(val, out x)) {
					txtAmplitude.Text = x.ToString();
				}
			}

			if ((val = sr.Load("SuffixSides")) != null) {
				txtSuffixSides.Text = val;
			}

			if ((val = sr.Load("SuffixCenter")) != null) {
				txtSuffixCenter.Text = val;
			}

			if ((val = sr.Load("Format")) != null) {
				if (val == "FLAC" && rbFLAC.Visible) {
					rbFLAC.Checked = true;
				}
				else if (val == "WavPack" && rbWavPack.Visible) {
					rbWavPack.Checked = true;
				}
				else {
					rbWAV.Checked = true;
				}
			}
		}

		private void SaveSettings() {
			SettingsWriter sw = new SettingsWriter("Center Cut GUI", "settings.txt");
			string val;

			if (rbSides.Checked) {
				val = "Sides";
			}
			else if (rbCenter.Checked) {
				val = "Center";
			}
			else {
				val = "Both";
			}
			sw.Save("Channels", val);

			sw.Save("DualCenter", chkDualCenter.Checked ? "1" : "0");

			sw.Save("BassToSides", chkBassToSides.Checked ? "1" : "0");

			sw.Save("Amplitude", txtAmplitude.Text.Trim());

			sw.Save("SuffixSides", txtSuffixSides.Text);

			sw.Save("SuffixCenter", txtSuffixCenter.Text);

			if (rbFLAC.Checked) {
				val = "FLAC";
			}
			else if (rbWavPack.Checked) {
				val = "WavPack";
			}
			else {
				val = "WAV";
			}
			sw.Save("Format", val);

			sw.Close();
		}

		private void btnRun_Click(object sender, EventArgs e) {
			if (btnRun.Text == "Run") {
				ProcessFilesParameters p;
				int pathCount;
				Thread workThread;

				btnRun.Text = "Stop";
				_stopFlag = false;

				pathCount = lstFiles.Items.Count;
				p.Paths = new string[pathCount];
				for (int i = 0; i < pathCount; i++) {
					p.Paths[i] = (string)lstFiles.Items[i];
				}
				p.OutputSides = rbSides.Checked | rbBoth.Checked;
				p.OutputCenter = rbCenter.Checked | rbBoth.Checked;
				p.SuffixSides = txtSuffixSides.Text.Trim();
				p.SuffixCenter = txtSuffixCenter.Text.Trim();
				p.DualCenter = chkDualCenter.Checked;
				p.BassToSides = chkBassToSides.Checked;
				try {
					p.AmpFactor = Double.Parse(txtAmplitude.Text.Trim()) / 100.0;
				}
				catch {
					MessageBox.Show("Invalid amplitude value.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return;
				}
				if (rbFLAC.Checked) {
					p.FormatExt = ".flac";
				}
				else if (rbWavPack.Checked) {
					p.FormatExt = ".wv";
				}
				else {
					p.FormatExt = ".wav";
				}

				workThread = new Thread(ProcessFiles);
				workThread.Start((object)p);
			}
			else {
				_stopFlag = true;
			}
		}

		private void btnAbout_Click(object sender, EventArgs e) {
			MessageBox.Show(String.Format("Center Cut GUI v1.6.0{0}Copyright 2006-2010 J.D. Purcell{0}" +
				"http://www.moitah.net/{0}{0}Center Cut algorithm by Avery Lee{0}FFT library by Takuya Ooura",
				Environment.NewLine), "About", MessageBoxButtons.OK, MessageBoxIcon.Information);
		}

		private void frmCenterCutGUI_Load(object sender, EventArgs e) {
			LoadSettings();
		}

		private void frmCenterCutGUI_FormClosed(object sender, FormClosedEventArgs e) {
			SaveSettings();
		}

		private void lstFiles_DragEnter(object sender, DragEventArgs e) {
			if (e.Data.GetDataPresent(DataFormats.FileDrop)) {
				e.Effect = DragDropEffects.Copy;
			}
		}

		private void lstFiles_DragDrop(object sender, DragEventArgs e) {
			if (e.Data.GetDataPresent(DataFormats.FileDrop)) {
				string[] drops = (string[])e.Data.GetData(DataFormats.FileDrop);

				for (int i = 0; i < drops.Length; i++) {
					string ext = Path.GetExtension(drops[i]).ToLower(CultureInfo.InvariantCulture);
					if ((ext == ".wav") ||
						(ext == ".flac" && rbFLAC.Visible) ||
						(ext == ".wv" && rbWavPack.Visible))
					{
						lstFiles.Items.Add(drops[i]);
					}
				}
			}
		}

		private void btnAdd_Click(object sender, EventArgs e) {
			OpenFileDialog fileDlg = new OpenFileDialog();
			DialogResult dlgRes;

			fileDlg.Title = "Input Audio";
			fileDlg.Multiselect = true;

			string[] readerExts = AudioReadWrite.ReaderExtensions;
			for (int i = 0; i < readerExts.Length; i++) {
				readerExts[i] = "*" + readerExts[i];
			}
			fileDlg.Filter = String.Format("Audio Files ({0})|{1}", String.Join("; ", readerExts), String.Join(";", readerExts));

			dlgRes = fileDlg.ShowDialog();
			if (dlgRes == DialogResult.OK) {
				string[] paths = fileDlg.FileNames;
				Array.Sort(paths);
				lstFiles.Items.AddRange(paths);
			}
		}

		private void btnClear_Click(object sender, EventArgs e) {
			lstFiles.Items.Clear();
		}

		private void btnRemove_Click(object sender, EventArgs e) {
			int s = lstFiles.SelectedIndex;
			int count = lstFiles.Items.Count - 1;
			if (s != -1) {
				lstFiles.Items.RemoveAt(s);
				if (count != 0) {
					lstFiles.SelectedIndex = Math.Min(s, count - 1);
				}
			}
		}

		private void btnUp_Click(object sender, EventArgs e) {
			int s = lstFiles.SelectedIndex;
			if ((s != -1) && (s != 0)) {
				ListBoxSwapItems(lstFiles, s, s - 1);
				lstFiles.SelectedIndex = s - 1;
			}
		}

		private void btnDown_Click(object sender, EventArgs e) {
			int s = lstFiles.SelectedIndex;
			if ((s != -1) && (s != lstFiles.Items.Count - 1)) {
				ListBoxSwapItems(lstFiles, s, s + 1);
				lstFiles.SelectedIndex = s + 1;
			}
		}
	}

	struct ProcessFilesParameters {
		public string[] Paths;
		public bool OutputSides;
		public bool OutputCenter;
		public string SuffixSides;
		public string SuffixCenter;
		public string FormatExt;
		public bool DualCenter;
		public bool BassToSides;
		public double AmpFactor;
	}
}
