﻿// ****************************************************************************
// 
// Center Cut GUI
// Copyright (C) 2006-2010 J.D. Purcell
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************

using System;
using System.Collections.Generic;
using System.Threading;

namespace JDP {
	delegate void TaskWork();

	class Task {
		ManualResetEvent _finished;
		TaskWork _action;

		public Task(TaskWork action) {
			_finished = new ManualResetEvent(false);
			_action = action;
			TaskScheduler.Schedule(this);
		}

		public void Execute() {
			if (_action == null) throw new Exception("Task already exectued.");
			_action();
			_action = null;
			_finished.Set();
		}

		public void Wait() {
			_finished.WaitOne();
		}
	}

	static class TaskScheduler {
		private static AutoResetEvent _taskQueued;
		private static Thread[] _threads;
		private static Queue<Task> _tasks;
		
		static TaskScheduler() {
			_taskQueued = new AutoResetEvent(false);
			_tasks = new Queue<Task>();
			ThreadCount = Math.Min(Environment.ProcessorCount, 32);
		}

		public static int ThreadCount { get; set; }

		public static void Schedule(Task task) {
			lock (_tasks) {
				_tasks.Enqueue(task);
				_taskQueued.Set();
				if (_threads == null) {
					_threads = new Thread[ThreadCount];
					for (int i = 0; i < _threads.Length; i++) {
						_threads[i] = new Thread(ThreadProc);
						_threads[i].IsBackground = true;
						_threads[i].Start();
					}
				}
			}
		}

		private static void ThreadProc() {
			while (true) {
				Task task = null;
				lock (_tasks) {
					if (_tasks.Count != 0) {
						task = _tasks.Dequeue();
					}
				}
				if (task != null) {
					task.Execute();
				}
				else {
					_taskQueued.WaitOne();
				}
			}
		}
	}
}
