// ****************************************************************************
// 
// Center Cut GUI
// Copyright (C) 2006-2010 J.D. Purcell
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************

using System;
using System.Runtime.InteropServices;

namespace JDP {
	public enum SampleConvertType {
		BytesToDouble,
		DoubleToBytes
	}

	public static class SampleHelper {
		public static unsafe void ConvertSamples(SampleConvertType type, byte[] samplesB, int offsetB,
			double[,] samplesD, int offsetD, int sampleCount, int bitsPerSample, int chanCount)
		{
			const double SampleScaleInv = 2147483648.0;
			const double SampleScale = 1.0 / SampleScaleInv;
			const double SampleMin = -2147483648.0;
			const double SampleMax = 2147483647.0;

			int bytesPerSample, shiftCount, xor;

			bytesPerSample = (bitsPerSample + 7) / 8;
			shiftCount = (4 - bytesPerSample) * 8;
			xor = (bytesPerSample == 1) ? (1 << 31) : 0;

			fixed (double* samplesDFixed = samplesD) {
				fixed (byte* samplesBFixed = samplesB) {
					double* sampD = samplesDFixed + (offsetD * chanCount);
					byte* sampB = samplesBFixed + (offsetB * chanCount * bytesPerSample);
					byte* max = sampB + (sampleCount * bytesPerSample * chanCount);

					if (type == SampleConvertType.BytesToDouble) {
						int tempI;

						while (sampB < max) {
							tempI = (*((int*)sampB) << shiftCount) ^ xor;
							*sampD = (double)tempI * SampleScale;

							sampB += bytesPerSample;
							sampD += 1;
						}
					}
					else {
						byte* maxw = max - 3;
						double tempD;
						uint tempI;

						while (sampB < max) {
							tempD = *sampD * SampleScaleInv;
							if (tempD > 0.0) {
								if (tempD > SampleMax) {
									tempD = SampleMax;
								}
								tempD += 0.5;
							}
							else {
								if (tempD < SampleMin) {
									tempD = SampleMin;
								}
								tempD -= 0.5;
							}
							tempI = (uint)((int)tempD ^ xor) >> shiftCount;

							if (sampB < maxw) {
								*((uint*)sampB) = tempI;
							}
							else {
								Marshal.Copy(BitConverter.GetBytes(tempI), 0, (IntPtr)sampB, bytesPerSample);
							}

							sampB += bytesPerSample;
							sampD += 1;
						}
					}
				}
			}
		}

		public static unsafe void MonoToStereoInPlace(byte[] buff, int sampleCount) {
			fixed (byte* buffFixed = buff) {
				short* pM = (short*)buffFixed + (sampleCount - 1);
				short* pS = (short*)buffFixed + ((sampleCount - 1) * 2);

				while (sampleCount-- > 0) {
					pS[1] = pM[0];
					pS[0] = pM[0];
					pM -= 1;
					pS -= 2;
				}
			}
		}
	}
}
