﻿// ****************************************************************************
// 
// Center Cut GUI
// Copyright (C) 2006-2010 J.D. Purcell
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************

using System;

namespace JDP {
	public static class CenterCutHelper {
		public delegate bool Progress(double percent);

		public static void RunCenterCut(string pathSrc, string pathDstSides, string pathDstCenter, bool dualCenter, bool bassToSides, double ampFactor, Progress progress) {
			const int updateInterval = 50000;
			IAudioSource aSrc;
			IAudioDest aDstS, aDstC;
			CenterCut centerCut;
			byte[] bytesSrc, bytesDst;
			int updateDelay, buffLength, inputLength;

			aSrc = AudioReadWrite.GetAudioReader(pathSrc);
			if (aSrc.ChannelCount != 2) {
				throw new Exception("Input audio must be stereo.");
			}
			if ((aSrc.BitsPerSample % 8) != 0) {
				throw new Exception("Input audio's bit depth must be a multiple of 8.");
			}
			aDstS = null;
			aDstC = null;
			if (pathDstSides != null) {
				aDstS = AudioReadWrite.GetAudioWriter(pathDstSides, aSrc.BitsPerSample, 2, aSrc.SampleRate, aSrc.Length);
			}
			if (pathDstCenter != null) {
				aDstC = AudioReadWrite.GetAudioWriter(pathDstCenter, aSrc.BitsPerSample, dualCenter ? 2 : 1, aSrc.SampleRate, aSrc.Length);
			}
			centerCut = new CenterCut(aSrc.SampleRate, 8192, ampFactor, bassToSides);
			updateDelay = 0;
			buffLength = centerCut.BlockSize;
			bytesSrc = new byte[buffLength * (aSrc.BitsPerSample / 8) * 2];
			bytesDst = new byte[bytesSrc.Length];

			centerCut.OnOutput = (count, sides, center) => {
				if (aDstS != null) {
					SampleHelper.ConvertSamples(SampleConvertType.DoubleToBytes,
						bytesDst, 0, sides, 0, count, aSrc.BitsPerSample, 2);

					aDstS.Write(bytesDst, count);
				}

				if (aDstC != null) {
					SampleHelper.ConvertSamples(SampleConvertType.DoubleToBytes,
						bytesDst, 0, center, 0, count, aSrc.BitsPerSample, 1);

					if (dualCenter) {
						SampleHelper.MonoToStereoInPlace(bytesDst, count);
					}

					aDstC.Write(bytesDst, count);
				}
			};

			while (aSrc.Remaining > 0) {
				if (updateDelay <= 0) {
					if (progress != null) {
						bool stop = progress((double)(aSrc.Length - aSrc.Remaining) / (double)aSrc.Length);
						if (stop) break;
					}
					updateDelay = updateInterval;
				}

				inputLength = (int)Math.Min((long)buffLength, aSrc.Remaining);

				aSrc.Read(bytesSrc, inputLength);

				double[,] inputBuffer = centerCut.GetInputBlock();

				SampleHelper.ConvertSamples(SampleConvertType.BytesToDouble,
					bytesSrc, 0, inputBuffer, 0, inputLength, aSrc.BitsPerSample, 2);

				centerCut.ProcessBlock(inputLength);

				updateDelay -= inputLength;
			}

			centerCut.FinishProcessing();

			aSrc.Close();
			if (aDstS != null) try { aDstS.Close(); } catch { }
			if (aDstC != null) try { aDstC.Close(); } catch { }
		}
	}
}
