// ****************************************************************************
// 
// Center Cut GUI
// Copyright (C) 2006-2010 J.D. Purcell
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************

using System;
using System.Collections.Generic;

namespace JDP {
	public delegate void CenterCutOutput(int count, double[,] sides, double[,] center);

	public class CenterCut {
		private static readonly double[] _powerIntegrals = new double[] {
			   1.0 /    1.0,
			   1.0 /    2.0,
			   3.0 /    8.0,
			   5.0 /   16.0,
			  35.0 /  128.0,
			  63.0 /  256.0,
			 231.0 / 1024.0,
			 429.0 / 2048.0
		};

		private CenterCutOutput _outputCallback;
		private int _windowSize;
		private int _overlapSize;
		private int _overlapCount;
		private int _outputWindowPower;
		private int _sampleRate;
		private double _ampFactor;
		private bool _bassToSides;
		private double[] _inputWindow;
		private double[] _outputWindow;
		private int _taskCount;
		private List<Task> _tasks;
		private List<RealFFT> _ffts;
		private List<double[,]> _inputs;
		private List<int> _inputCounts;
		private List<double[]> _outputs;
		private List<double[]> _fftBuffers;
		private double[,] _outputSides;
		private double[,] _outputCenter;

		public CenterCut(int sampleRate, int windowSize, double ampFactor, bool bassToSides) {
			_sampleRate = sampleRate;
			_windowSize = windowSize;
			_overlapCount = 4;
			_outputWindowPower = 2;
			_ampFactor = ampFactor;
			_bassToSides = bassToSides;
			_taskCount = TaskScheduler.ThreadCount;

			Init();
		}

		public CenterCut(int sampleRate) : this(sampleRate, 8192, 1.0, false) {
		}

		public CenterCutOutput OnOutput {
			get {
				return _outputCallback;
			}
			set {
				_outputCallback = value;
			}
		}

		public int WindowSize {
			get {
				return _windowSize;
			}
		}

		public int BlockSize {
			get {
				return _overlapSize;
			}
		}

		public double[,] GetInputBlock() {
			if (_tasks[0] != null) _tasks[0].Wait();

			Array.Clear(_inputs[0], 0, _overlapSize * 2);

			return _inputs[0];
		}

		public void ProcessBlock(int sampleCount) {
			RealFFT fft = _ffts[0];
			double[,] input = _inputs[0];
			double[] output = _outputs[0];
			double[] fftBuffL = _fftBuffers[0];
			double[] fftBuffR = _fftBuffers[1];
			double[] fftBuffC = _fftBuffers[2];

			Array.Clear(output, 0, _overlapSize);

			_tasks.RemoveAt(0);
			_ffts.RemoveAt(0);
			_inputs.RemoveAt(0);
			_inputCounts.RemoveAt(0);
			_outputs.RemoveAt(0);
			_fftBuffers.RemoveRange(0, 3);

			int iTask = _tasks.Count;
			double[][,] inputs = new double[_overlapCount][,];
			int[] inputCounts = new int[_overlapCount];
			double[][] outputs = new double[_overlapCount][];
			Task prevTask = (iTask != 0) ? _tasks[iTask - 1] : null;

			_ffts.Add(fft);
			_inputs.Add(input);
			_inputCounts.Add(sampleCount);
			_outputs.Add(output);
			_fftBuffers.Add(fftBuffL);
			_fftBuffers.Add(fftBuffR);
			_fftBuffers.Add(fftBuffC);

			for (int i = 0; i < _overlapCount; i++) {
				inputs[i] = _inputs[iTask + i];
				inputCounts[i] = _inputCounts[iTask + i];
				outputs[i] = _outputs[iTask + i];
			}

			_tasks.Add(
				new Task(() => {
					Run(fft, inputs, outputs, fftBuffL, fftBuffR, fftBuffC);
					if (prevTask != null) prevTask.Wait();

					if (inputCounts[0] != 0 && _outputCallback != null) {
						int count = inputCounts[0];
						for (int i = 0; i < count; i++) {
							double c = outputs[0][i];
							_outputCenter[i, 0] = c * _ampFactor;
							_outputSides[i, 0] = (inputs[0][i, 0] - c) * _ampFactor;
							_outputSides[i, 1] = (inputs[0][i, 1] - c) * _ampFactor;
						}
						_outputCallback(count, _outputSides, _outputCenter);
					}
				})
			);
		}

		public void FinishProcessing() {
			for (int i = 0; i < _overlapCount - 1; i++) {
				double[,] block = GetInputBlock();
				ProcessBlock(0);
			}

			while (_tasks.Count > 0) {
				_tasks[0].Wait();
				_tasks.RemoveAt(0);
			}
		}

		private void Init() {
			_overlapSize = _windowSize / _overlapCount;

			_tasks = new List<Task>();
			_ffts = new List<RealFFT>();
			_fftBuffers = new List<double[]>();
			for (int i = 0; i < _taskCount; i++) {
				_tasks.Add(null);
				_ffts.Add(new RealFFT(_windowSize));
				_fftBuffers.Add(new double[_windowSize]);
				_fftBuffers.Add(new double[_windowSize]);
				_fftBuffers.Add(new double[_windowSize]);
			}

			_inputs = new List<double[,]>();
			_inputCounts = new List<int>();
			_outputs = new List<double[]>();
			for (int i = 0; i < _overlapCount + (_taskCount - 1); i++) {
				_inputs.Add(new double[_overlapSize, 2]);
				_inputCounts.Add(0);
				_outputs.Add(new double[_overlapSize]);
			}

			_outputSides = new double[_overlapSize, 2];
			_outputCenter = new double[_overlapSize, 1];

			_inputWindow = CreateRaisedCosineWindow(_windowSize, 1.0);
			_outputWindow = CreateRaisedCosineWindow(_windowSize, (double)_outputWindowPower);
			for (int i = 0; i < _windowSize; i++) {
				_outputWindow[i] /= (_windowSize / 2.0) * _powerIntegrals[1 + _outputWindowPower] * _overlapCount;
			}
		}

		private void Run(RealFFT fft, double[][,] inputs, double[][] outputs, double[] buffL, double[] buffR, double[] buffC) {
			int freqBelowToSides = Convert.ToInt32(200.0 / ((double)_sampleRate / _windowSize)) * 2;

			// copy to temporary buffer, apply input window, and FFT

			for (int iBlock = 0; iBlock < _overlapCount; iBlock++) {
				for (int iBlockSample = 0; iBlockSample < _overlapSize; iBlockSample++) {
					int iBuffer = (iBlock * _overlapSize) + iBlockSample;
					double w = _inputWindow[iBuffer];

					buffL[iBuffer] = inputs[iBlock][iBlockSample, 0] * w;
					buffR[iBuffer] = inputs[iBlock][iBlockSample, 1] * w;
				}
			}

			fft.ComputeForward(buffL);
			fft.ComputeForward(buffR);

			// perform stereo separation

			buffC[0] = 0;
			buffC[1] = 0;
			for (int i = 2; i < _windowSize; i += 2) {
				double lR = buffL[i    ];
				double lI = buffL[i + 1];
				double rR = buffR[i    ];
				double rI = buffR[i + 1];

				double sumR = lR + rR;
				double sumI = lI + rI;
				double diffR = lR - rR;
				double diffI = lI - rI;

				double sumSq = sumR * sumR + sumI * sumI;
				double diffSq = diffR * diffR + diffI * diffI;
				double alpha = 0.0;

				if (sumSq > 0.000000000000001) {
					alpha = 0.5 - Math.Sqrt(diffSq / sumSq) * 0.5;
				}

				double cR = sumR * alpha;
				double cI = sumI * alpha;

				if (_bassToSides && (i < freqBelowToSides)) {
					cR = cI = 0.0;
				}

				buffC[i    ] = cR;
				buffC[i + 1] = cI;
			}

			// reconstitute left/right/center channels

			fft.ComputeReverse(buffC);

			// apply output window

			for (int i = 0; i < _windowSize; i++) {
				buffC[i] *= _outputWindow[i];
			}

			// writeout

			for (int iBlock = 0; iBlock < _overlapCount; iBlock++) {
				lock (outputs[iBlock]) {
					for (int iBlockSample = 0; iBlockSample < _overlapSize; iBlockSample++) {
						int iBuffer = (iBlock * _overlapSize) + iBlockSample;

						outputs[iBlock][iBlockSample] += buffC[iBuffer];
					}
				}
			}
		}

		private double[] CreateRaisedCosineWindow(int n, double power) {
			double twopi_over_n = (Math.PI * 2) / n;
			double[] dst = new double[n];

			for (int i = 0; i < n; i++) {
				dst[i] = Math.Pow(0.5 * (1.0 - Math.Cos(twopi_over_n * (i + 0.5))), power);
			}

			return dst;
		}
	}
}
