﻿// ****************************************************************************
// 
// CUE Tools
// Copyright (C) 2006-2010 J.D. Purcell
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************

using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;

namespace JDP {
	public interface IAudioSource {
		int Read(byte[] buff, int sampleCount);
		long Length { get; }
		long Position { get; set; }
		long Remaining { get; }
		void Close();
		int BitsPerSample { get; }
		int ChannelCount { get; }
		int SampleRate { get; }
	}

	public interface IAudioDest {
		void Write(byte[] buff, int sampleCount);
		void Close();
		long FinalSampleCount { set; }
	}

	public static class AudioReadWrite {
		private static Dictionary<string, string> _audioReaderClasses;
		private static Dictionary<string, string> _audioWriterClasses;
		private static List<string> _audioReaderExtensions;
		private static List<string> _audioWriterExtensions;

		static AudioReadWrite() {
			_audioReaderClasses = new Dictionary<string, string>(StringComparer.InvariantCultureIgnoreCase);
			_audioWriterClasses = new Dictionary<string, string>(StringComparer.InvariantCultureIgnoreCase);
			_audioReaderExtensions = new List<string>();
			_audioWriterExtensions = new List<string>();

			RegisterAudioReader(".wav", "WAVReader");
#if PLATFORM_X86
			RegisterAudioReader(".flac", "FLACReader");
			RegisterAudioReader(".wv", "WavPackReader");
#endif

			RegisterAudioWriter(".wav", "WAVWriter");
#if PLATFORM_X86
			RegisterAudioWriter(".flac", "FLACWriter");
			RegisterAudioWriter(".wv", "WavPackWriter");
#endif
		}

		private static void RegisterAudioReader(string extension, string className) {
			_audioReaderClasses.Add(extension, className);
			_audioReaderExtensions.Add(extension);
		}

		private static void RegisterAudioWriter(string extension, string className) {
			_audioWriterClasses.Add(extension, className);
			_audioWriterExtensions.Add(extension);
		}

		public static string[] ReaderExtensions {
			get { return _audioReaderExtensions.ToArray(); }
		}

		public static string[] WriterExtensions {
			get { return _audioWriterExtensions.ToArray(); }
		}

		public static IAudioSource GetAudioReader(string path) {
			if (path == "stdin") {
				return new WAVReader(Console.OpenStandardInput());
			}
			else {
				Type classType = GetAudioReadWriteType(_audioReaderClasses, Path.GetExtension(path));
				try {
					return (IAudioSource)Activator.CreateInstance(classType, new object[] { path });
				}
				catch (Exception ex) {
					throw ex.InnerException ?? ex;
				}
			}
		}

		public static IAudioDest GetAudioWriter(string path, int bitsPerSample, int channelCount, int sampleRate, long finalSampleCount) {
			IAudioDest dest;
			if (path == "stdout") {
				dest = new WAVWriter(Console.OpenStandardOutput(), bitsPerSample, channelCount, sampleRate);
			}
			else {
				Type classType = GetAudioReadWriteType(_audioWriterClasses, Path.GetExtension(path));
				try {
					dest = (IAudioDest)Activator.CreateInstance(classType, new object[] { path, bitsPerSample, channelCount, sampleRate });
				}
				catch (Exception ex) {
					throw ex.InnerException ?? ex;
				}
			}
			dest.FinalSampleCount = finalSampleCount;
			return dest;
		}

		private static Type GetAudioReadWriteType(Dictionary<string, string> classDict, string extension) {
			string className;
			Type classType = null;
			if (classDict.TryGetValue(extension, out className)) {
				classType = Assembly.GetExecutingAssembly().GetType("JDP." + className);
			}
			if (classType == null) {
				throw new Exception("Unsupported audio type.");
			}
			return classType;
		}
	}

	public class WAVReader : IAudioSource {
		BinaryReader _br;
		bool _canSeek;
		long _streamOffset, _streamLength;
		long _dataOffset, _dataLen;
		long _samplePos, _sampleLen;
		int _bitsPerSample, _channelCount, _sampleRate, _blockAlign;
		bool _largeFile;

		public WAVReader(string path) :
			this(new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.Read))
		{
		}

		public WAVReader(Stream stream) {
			_br = new BinaryReader(stream);
			_canSeek = stream.CanSeek;
			if (_canSeek) {
				_streamLength = stream.Length;
			}

			ParseHeaders();
		}

		public void Close() {
			_br.Close();
			_br = null;
		}

		private void ParseHeaders() {
			const long maxFileSize = 0x7FFFFFFEL;
			const uint fccRIFF = 0x46464952;
			const uint fccWAVE = 0x45564157;
			const uint fccFormat = 0x20746D66;
			const uint fccData = 0x61746164;

			uint lenRIFF;
			long fileEnd;
			bool foundFormat, foundData;

			if (ReadUInt32() != fccRIFF) {
				throw new Exception("Not a valid RIFF file.");
			}

			lenRIFF = ReadUInt32();
			fileEnd = (long)lenRIFF + 8;

			if (ReadUInt32() != fccWAVE) {
				throw new Exception("Not a valid WAVE file.");
			}

			_largeFile = false;
			foundFormat = false;
			foundData = false;

			while (true) {
				uint ckID, ckSize, ckSizePadded;
				long ckEnd;

				ckID = ReadUInt32();
				ckSize = ReadUInt32();
				ckSizePadded = (ckSize + 1U) & ~1U;
				ckEnd = _streamOffset + (long)ckSizePadded;

				if (ckID == fccFormat) {
					foundFormat = true;

					if (ReadUInt16() != 1) {
						throw new Exception("WAVE must be PCM format.");
					}
					_channelCount = ReadInt16();
					_sampleRate = ReadInt32();
					ReadInt32();
					_blockAlign = ReadInt16();
					_bitsPerSample = ReadInt16();
				}
				else if (ckID == fccData) {
					foundData = true;

					_dataOffset = _streamOffset;
					if (!_canSeek || _streamLength <= maxFileSize) {
						_dataLen = (long)ckSize;
					}
					else {
						_largeFile = true;
						_dataLen = _streamLength - _dataOffset;
					}

					if ((!_canSeek || _largeFile) && !foundFormat) {
						throw new Exception("Data chunk cannot appear before format chunk.");
					}
				}

				if (foundFormat && foundData) {
					break;
				}

				SkipBytes(ckEnd - _streamOffset);
			}

			if (_channelCount <= 0) {
				throw new Exception("Channel count is invalid.");
			}
			if (_sampleRate <= 0) {
				throw new Exception("Sample rate is invalid.");
			}
			if (_blockAlign != (_channelCount * ((_bitsPerSample + 7) / 8))) {
				throw new Exception("Block align is invalid.");
			}
			if ((_bitsPerSample <= 0) || (_bitsPerSample > 32)) {
				throw new Exception("Bits per sample is invalid.");
			}

			_sampleLen = _dataLen / (long)_blockAlign;

			if (_streamOffset != _dataOffset) {
				Position = 0;
			}
		}

		private short ReadInt16() {
			_streamOffset += 2;
			return _br.ReadInt16();
		}

		private ushort ReadUInt16() {
			_streamOffset += 2;
			return _br.ReadUInt16();
		}

		private int ReadInt32() {
			_streamOffset += 4;
			return _br.ReadInt32();
		}

		private uint ReadUInt32() {
			_streamOffset += 4;
			return _br.ReadUInt32();
		}

		private void SkipBytes(long count) {
			const int buffSize = 65536;
			if (count == 0) return;
			_streamOffset += count;
			if (!_canSeek || count <= buffSize) {
				byte[] buff = new byte[buffSize];
				while (count > 0) {
					count -= _br.Read(buff, 0, (int)Math.Min(count, buffSize));
				}
			}
			else {
				_br.BaseStream.Seek(count, SeekOrigin.Current);
			}
		}

		public long Position {
			get {
				return _samplePos;
			}
			set {
				if (!_canSeek) {
					throw new Exception("Input stream does not support seeking.");
				}

				if (value < 0) {
					_samplePos = 0;
				}
				else if (value > _sampleLen) {
					_samplePos = _sampleLen;
				}
				else {
					_samplePos = value;
				}

				_streamOffset = _dataOffset + (_samplePos * (long)_blockAlign);
				_br.BaseStream.Seek(_streamOffset, SeekOrigin.Begin);
			}
		}

		public long Length {
			get {
				return _sampleLen;
			}
		}

		public long Remaining {
			get {
				return _sampleLen - _samplePos;
			}
		}

		public int ChannelCount {
			get {
				return _channelCount;
			}
		}

		public int SampleRate {
			get {
				return _sampleRate;
			}
		}

		public int BitsPerSample {
			get {
				return _bitsPerSample;
			}
		}

		public int BlockAlign {
			get {
				return _blockAlign;
			}
		}

		public int Read(byte[] buff, int sampleCount) {
			int byteCount;

			if (sampleCount < 0) {
				sampleCount = 0;
			}
			else if (sampleCount > Remaining) {
				sampleCount = (int)Remaining;
			}
			byteCount = sampleCount * _blockAlign;

			if (sampleCount != 0) {
				int offset = 0;
				while (offset < byteCount) {
					offset += _br.Read(buff, offset, byteCount - offset);
				}
				_streamOffset += byteCount;
				_samplePos += sampleCount;
			}

			return sampleCount;
		}
	}

	public class WAVWriter : IAudioDest {
		BinaryWriter _bw;
		bool _canSeek;
		bool _wroteHeaders;
		int _bitsPerSample, _channelCount, _sampleRate, _blockAlign;
		long _finalSampleLen, _sampleLen;

		public WAVWriter(string path, int bitsPerSample, int channelCount, int sampleRate) :
			this(new FileStream(path, FileMode.Create, FileAccess.Write, FileShare.Read),
				bitsPerSample, channelCount, sampleRate)
		{
		}

		public WAVWriter(Stream stream, int bitsPerSample, int channelCount, int sampleRate) {
			_bitsPerSample = bitsPerSample;
			_channelCount = channelCount;
			_sampleRate = sampleRate;
			_blockAlign = _channelCount * ((_bitsPerSample + 7) / 8);

			_bw = new BinaryWriter(stream);
			_canSeek = stream.CanSeek;
		}

		private void WriteHeaders() {
			const uint fccRIFF = 0x46464952;
			const uint fccWAVE = 0x45564157;
			const uint fccFormat = 0x20746D66;
			const uint fccData = 0x61746164;

			uint dataChunkSize = GetDataChunkSize(_finalSampleLen);

			_bw.Write(fccRIFF);
			_bw.Write((uint)(dataChunkSize + (dataChunkSize & 1) + 36));
			_bw.Write(fccWAVE);

			_bw.Write(fccFormat);
			_bw.Write((uint)16);
			_bw.Write((ushort)1);
			_bw.Write((ushort)_channelCount);
			_bw.Write((uint)_sampleRate);
			_bw.Write((uint)(_sampleRate * _blockAlign));
			_bw.Write((ushort)_blockAlign);
			_bw.Write((ushort)_bitsPerSample);

			_bw.Write(fccData);
			_bw.Write((uint)dataChunkSize);
		}

		private uint GetDataChunkSize(long sampleCount) {
			const long maxFileSize = 0x7FFFFFFEL;
			long dataSize = sampleCount * _blockAlign;
			if ((dataSize + 44) > maxFileSize) {
				dataSize = ((maxFileSize - 44) / _blockAlign) * _blockAlign;
			}
			return (uint)dataSize;
		}

		public void Close() {
			if (((_sampleLen * _blockAlign) & 1) == 1) {
				_bw.Write((byte)0);
			}

			try {
				if (_sampleLen != _finalSampleLen) {
					if (_canSeek) {
						uint dataChunkSize = GetDataChunkSize(_sampleLen);
						_bw.Seek(4, SeekOrigin.Begin);
						_bw.Write((uint)(dataChunkSize + (dataChunkSize & 1) + 36));
						_bw.Seek(40, SeekOrigin.Begin);
						_bw.Write((uint)dataChunkSize);
					}
					else {
						throw new Exception("Samples written differs from the expected sample count.");
					}
				}
			}
			finally {
				_bw.Close();
				_bw = null;
			}
		}

		public long Position {
			get {
				return _sampleLen;
			}
		}

		public long FinalSampleCount {
			set {
				_finalSampleLen = value;
			}
		}

		public void Write(byte[] buff, int sampleCount) {
			if (sampleCount <= 0) return;

			if (!_wroteHeaders) {
				WriteHeaders();
				_wroteHeaders = true;
			}

			_bw.Write(buff, 0, sampleCount * _blockAlign);
			_sampleLen += sampleCount;
		}
	}

#if PLATFORM_X86
	class FLACReader : IAudioSource {
		FLACDotNet.FLACReader _flacReader;
		int[,] _sampleBuffer;
		int _bufferOffset, _bufferLength;

		public FLACReader(string path) {
			_flacReader = new FLACDotNet.FLACReader(path);
			_bufferOffset = 0;
			_bufferLength = 0;
		}

		public void Close() {
			_flacReader.Close();
		}

		public long Length {
			get {
				return _flacReader.Length;
			}
		}

		public long Remaining {
			get {
				return _flacReader.Remaining + SamplesInBuffer;
			}
		}

		public long Position {
			get {
				return _flacReader.Position - SamplesInBuffer;
			}
			set {
				_flacReader.Position = value;
				_bufferOffset = 0;
				_bufferLength = 0;
			}
		}

		private int SamplesInBuffer {
			get {
				return _bufferLength - _bufferOffset;
			}
		}

		public int BitsPerSample {
			get {
				return _flacReader.BitsPerSample;
			}
		}

		public int ChannelCount {
			get {
				return _flacReader.ChannelCount;
			}
		}

		public int SampleRate {
			get {
				return _flacReader.SampleRate;
			}
		}

		private unsafe void FLACSamplesToBytes_16(int[,] inSamples, int inSampleOffset,
			byte[] outSamples, int outByteOffset, int sampleCount, int channelCount)
		{
			int loopCount = sampleCount * channelCount;

			if ((inSamples.GetLength(0) - inSampleOffset < sampleCount) ||
				(outSamples.Length - outByteOffset < loopCount * 2))
			{
				throw new IndexOutOfRangeException();
			}

			fixed (int* pInSamplesFixed = &inSamples[inSampleOffset, 0]) {
				fixed (byte* pOutSamplesFixed = &outSamples[outByteOffset]) {
					int* pInSamples = pInSamplesFixed;
					short* pOutSamples = (short*)pOutSamplesFixed;

					for (int i = 0; i < loopCount; i++) {
						*(pOutSamples++) = (short)*(pInSamples++);
					}
				}
			}
		}

		public int Read(byte[] buff, int sampleCount) {
			if (_flacReader.BitsPerSample != 16) {
				throw new Exception("Reading is only supported for 16 bit sample depth.");
			}
			int chanCount = _flacReader.ChannelCount;
			int samplesNeeded, copyCount, buffOffset;

			buffOffset = 0;
			samplesNeeded = sampleCount;

			while (samplesNeeded != 0) {
				if (SamplesInBuffer == 0) {
					_bufferOffset = 0;
					_bufferLength = _flacReader.Read(out _sampleBuffer);
				}

				copyCount = Math.Min(samplesNeeded, SamplesInBuffer);

				FLACSamplesToBytes_16(_sampleBuffer, _bufferOffset, buff, buffOffset,
					copyCount, chanCount);

				samplesNeeded -= copyCount;
				buffOffset += copyCount * chanCount * 2;
				_bufferOffset += copyCount;
			}

			return sampleCount;
		}
	}

	class FLACWriter : IAudioDest {
		FLACDotNet.FLACWriter _flacWriter;
		int[,] _sampleBuffer;
		int _bitsPerSample;
		int _channelCount;
		int _sampleRate;

		public FLACWriter(string path, int bitsPerSample, int channelCount, int sampleRate) {
			if (bitsPerSample != 16) {
				throw new Exception("Bits per sample must be 16.");
			}
			_bitsPerSample = bitsPerSample;
			_channelCount = channelCount;
			_sampleRate = sampleRate;
			_flacWriter = new FLACDotNet.FLACWriter(path, bitsPerSample, channelCount, sampleRate);
		}

		public long FinalSampleCount {
			get {
				return _flacWriter.FinalSampleCount;
			}
			set {
				_flacWriter.FinalSampleCount = value;
			}
		}

		public int CompressionLevel {
			get {
				return _flacWriter.CompressionLevel;
			}
			set {
				_flacWriter.CompressionLevel = value;
			}
		}

		public bool Verify {
			get {
				return _flacWriter.Verify;
			}
			set {
				_flacWriter.Verify = value;
			}
		}

		public void Close() {
			_flacWriter.Close();
		}

		private unsafe void BytesToFLACSamples_16(byte[] inSamples, int inByteOffset,
			int[,] outSamples, int outSampleOffset, int sampleCount, int channelCount)
		{
			int loopCount = sampleCount * channelCount;

			if ((inSamples.Length - inByteOffset < loopCount * 2) ||
				(outSamples.GetLength(0) - outSampleOffset < sampleCount))
			{
				throw new IndexOutOfRangeException();
			}

			fixed (byte* pInSamplesFixed = &inSamples[inByteOffset]) {
				fixed (int* pOutSamplesFixed = &outSamples[outSampleOffset, 0]) {
					short* pInSamples = (short*)pInSamplesFixed;
					int* pOutSamples = pOutSamplesFixed;

					for (int i = 0; i < loopCount; i++) {
						*(pOutSamples++) = (int)*(pInSamples++);
					}
				}
			}
		}

		public void Write(byte[] buff, int sampleCount) {
			if ((_sampleBuffer == null) || (_sampleBuffer.GetLength(0) < sampleCount)) {
				_sampleBuffer = new int[sampleCount, _channelCount];
			}
			BytesToFLACSamples_16(buff, 0, _sampleBuffer, 0, sampleCount, _channelCount);
			_flacWriter.Write(_sampleBuffer, sampleCount);
		}
	}

	class WavPackReader : IAudioSource {
		WavPackDotNet.WavPackReader _wavPackReader;

		public WavPackReader(string path) {
			_wavPackReader = new WavPackDotNet.WavPackReader(path);
		}

		public void Close() {
			_wavPackReader.Close();
		}

		public long Length {
			get {
				return _wavPackReader.Length;
			}
		}

		public long Remaining {
			get {
				return _wavPackReader.Remaining;
			}
		}

		public long Position {
			get {
				return _wavPackReader.Position;
			}
			set {
				_wavPackReader.Position = (int)value;
			}
		}

		public int BitsPerSample {
			get {
				return _wavPackReader.BitsPerSample;
			}
		}

		public int ChannelCount {
			get {
				return _wavPackReader.ChannelCount;
			}
		}

		public int SampleRate {
			get {
				return _wavPackReader.SampleRate;
			}
		}

		private unsafe void WavPackSamplesToBytes_16(int[,] inSamples, int inSampleOffset,
			byte[] outSamples, int outByteOffset, int sampleCount, int channelCount)
		{
			int loopCount = sampleCount * channelCount;

			if ((inSamples.GetLength(0) - inSampleOffset < sampleCount) ||
				(outSamples.Length - outByteOffset < loopCount * 2))
			{
				throw new IndexOutOfRangeException();
			}

			fixed (int* pInSamplesFixed = &inSamples[inSampleOffset, 0]) {
				fixed (byte* pOutSamplesFixed = &outSamples[outByteOffset]) {
					int* pInSamples = pInSamplesFixed;
					short* pOutSamples = (short*)pOutSamplesFixed;

					for (int i = 0; i < loopCount; i++) {
						*(pOutSamples++) = (short)*(pInSamples++);
					}
				}
			}
		}

		public int Read(byte[] buff, int sampleCount) {
			if (_wavPackReader.BitsPerSample != 16) {
				throw new Exception("Reading is only supported for 16 bit sample depth.");
			}
			int chanCount = _wavPackReader.ChannelCount;
			int[,] sampleBuffer;

			sampleBuffer = new int[sampleCount * 2, chanCount];
			_wavPackReader.Read(sampleBuffer, sampleCount);
			WavPackSamplesToBytes_16(sampleBuffer, 0, buff, 0, sampleCount, chanCount);

			return sampleCount;
		}
	}

	class WavPackWriter : IAudioDest {
		WavPackDotNet.WavPackWriter _wavPackWriter;
		int[,] _sampleBuffer;
		int _bitsPerSample;
		int _channelCount;
		int _sampleRate;

		public WavPackWriter(string path, int bitsPerSample, int channelCount, int sampleRate) {
			if (bitsPerSample != 16) {
				throw new Exception("Bits per sample must be 16.");
			}
			_bitsPerSample = bitsPerSample;
			_channelCount = channelCount;
			_sampleRate = sampleRate;
			_wavPackWriter = new WavPackDotNet.WavPackWriter(path, bitsPerSample, channelCount, sampleRate);
		}

		public long FinalSampleCount {
			get {
				return _wavPackWriter.FinalSampleCount;
			}
			set {
				_wavPackWriter.FinalSampleCount = (int)value;
			}
		}

		public int CompressionMode {
			get {
				return _wavPackWriter.CompressionMode;
			}
			set {
				_wavPackWriter.CompressionMode = value;
			}
		}

		public int ExtraMode {
			get {
				return _wavPackWriter.ExtraMode;
			}
			set {
				_wavPackWriter.ExtraMode = value;
			}
		}

		public void Close() {
			_wavPackWriter.Close();
		}

		private unsafe void BytesToWavPackSamples_16(byte[] inSamples, int inByteOffset,
			int[,] outSamples, int outSampleOffset, int sampleCount, int channelCount)
		{
			int loopCount = sampleCount * channelCount;

			if ((inSamples.Length - inByteOffset < loopCount * 2) ||
				(outSamples.GetLength(0) - outSampleOffset < sampleCount))
			{
				throw new IndexOutOfRangeException();
			}

			fixed (byte* pInSamplesFixed = &inSamples[inByteOffset]) {
				fixed (int* pOutSamplesFixed = &outSamples[outSampleOffset, 0]) {
					short* pInSamples = (short*)pInSamplesFixed;
					int* pOutSamples = pOutSamplesFixed;

					for (int i = 0; i < loopCount; i++) {
						*(pOutSamples++) = (int)*(pInSamples++);
					}
				}
			}
		}

		public void Write(byte[] buff, int sampleCount) {
			if ((_sampleBuffer == null) || (_sampleBuffer.GetLength(0) < sampleCount)) {
				_sampleBuffer = new int[sampleCount, _channelCount];
			}
			BytesToWavPackSamples_16(buff, 0, _sampleBuffer, 0, sampleCount, _channelCount);
			_wavPackWriter.Write(_sampleBuffer, sampleCount);
		}
	}
#endif
}
