﻿// ****************************************************************************
// 
// Center Cut GUI
// Copyright (C) 2006-2010 J.D. Purcell
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// ****************************************************************************
using System;
using System.Diagnostics;
using System.IO;

namespace JDP {
	class Program {
		static bool _autoOverwrite;
		static bool _silent;

		static int Main(string[] args) {
			int argCount = args.Length;
			int argIndex = 0;
			string pathSrc = null;
			string pathDstSides = null;
			string pathDstCenter = null;
			double ampFactor = 1.0;
			bool dualCenter = false;
			bool bassToSides = false;
			int threadCount = 0;

			try {
				pathSrc = args[argIndex++];
				if (pathSrc.StartsWith("-") && !File.Exists(pathSrc)) {
					throw new Exception();
				}
				if (pathSrc == "stdin") {
					_silent = true;
				}

				while (argIndex < argCount) {
					switch (args[argIndex]) {
						case "-s":
							pathDstSides = args[++argIndex];
							if (pathDstSides == "stdout") {
								_silent = true;
								if (pathDstCenter != null) throw new Exception();
							}
							break;
						case "-c":
							pathDstCenter = args[++argIndex];
							if (pathDstCenter == "stdout") {
								_silent = true;
								if (pathDstSides != null) throw new Exception();
							}
							break;
						case "-a":
							ampFactor = Double.Parse(args[++argIndex]) / 100.0;
							break;
						case "-d":
							dualCenter = true;
							break;
						case "-b":
							bassToSides = true;
							break;
						case "-o":
							_autoOverwrite = true;
							break;
						case "-t":
							threadCount = Int32.Parse(args[++argIndex]);
							if (threadCount < 1 || threadCount > 32) throw new Exception();
							break;
						default:
							goto BreakArgLoop;
					}
					argIndex++;
				}
			BreakArgLoop:

				if (argIndex != argCount) {
					throw new Exception();
				}
			}
			catch {
				WriteAbout();
				WriteUsage();
				return 1;
			}

			if (!_silent) {
				WriteAbout();
			}

			if (pathDstSides != null && pathDstSides != "stdout" && File.Exists(pathDstSides) && !PromptOverwrite(pathDstSides)) return 0;
			if (pathDstCenter != null && pathDstCenter != "stdout" && File.Exists(pathDstCenter) && !PromptOverwrite(pathDstCenter)) return 0;

			if (threadCount != 0) {
				TaskScheduler.ThreadCount = threadCount;
			}

			try {
				Stopwatch st = new Stopwatch();
				st.Start();
				CenterCutHelper.RunCenterCut(pathSrc, pathDstSides, pathDstCenter, dualCenter, bassToSides, ampFactor, Progress);
				st.Stop();
				if (!_silent) Console.WriteLine("\rFinished in {0:0.000} seconds.", st.Elapsed.TotalSeconds);
				return 0;
			}
			catch (Exception ex) {
				Console.Error.WriteLine("Error: " + ex.Message);
				return 1;
			}
		}

		private static void WriteAbout() {
			Console.WriteLine("Center Cut CL v1.6.0");
			Console.WriteLine("Copyright 2006-2010 J.D. Purcell");
			Console.WriteLine("Center Cut algorithm by Avery Lee");
			Console.WriteLine("FFT library by Takuya Ooura");
			Console.WriteLine("http://www.moitah.net/");
			Console.WriteLine();
		}

		private static void WriteUsage() {
			Console.WriteLine("Arguments: source_file [switches]");
			Console.WriteLine();
			Console.WriteLine("Switches:");
			Console.WriteLine("  -s <file>     Write side channels to file.");
			Console.WriteLine("  -c <file>     Write center channel to file.");
			Console.WriteLine("  -a <percent>  Scale output amplitude by percent (default 100).");
			Console.WriteLine("  -d            Dual-mono center channel.");
			Console.WriteLine("  -b            Move bass to side channels.");
			Console.WriteLine("  -o            Overwrite output files without prompting.");
			Console.WriteLine();
			Console.WriteLine("\"stdin\" or \"stdout\" may be used in place of input or output files.");
		}

		private static bool Progress(double percent) {
			if (!_silent) Console.Write("\r{0:0}%", (int)(percent * 100.0));
			return false;
		}

		private static bool PromptOverwrite(string path) {
			if (_autoOverwrite) return true;
			if (_silent) {
				Console.Error.WriteLine("Error: Output file {0} already exists.", Path.GetFileName(path));
				return false;
			}
			bool? overwrite = null;
			Console.Write("Output file \"" + Path.GetFileName(path) + "\" already exists, overwrite? (y/n): ");
			while (overwrite == null) {
				char c = Console.ReadKey(true).KeyChar;
				if (c == 'y') overwrite = true;
				if (c == 'n') overwrite = false;
			}
			Console.WriteLine(overwrite.Value ? "y" : "n");
			Console.WriteLine();
			return overwrite.Value;
		}
	}
}
